/*****************************************************************************/
/* Betriebssysteme I, Uni-Magdeburg, SS 97; Uni Potsdam WS 99/00             */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*                          K E Y C T R L                                    */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Das Modul KEYCTRL, implementiert Funktionen, die fuer die Tastaturabfrage */
/* im Rahmen einer Unterbrechungsbehandlung und die Interpretation der       */
/* gesandten Codes benoetigt werden.                                         */
/*****************************************************************************/

/* INCLUDES */

#include "machine/ioport.h"
#include "machine/pic.h"
#include "machine/keyctrl.h"

/* DEFINES */

/* Bits im Statusregister */

#define OUTB            0x01  
#define INPB            0x02
#define AUXB            0x20

/* Kommandos des Tastaturcontrollers */

#define CMD_SET_SPEED   0xf3
#define CMD_RESET       0xfc

/* Konstanten fuer die Tastaturdekodierung */

#define BREAK           ((unsigned char) 0x80)
#define PREFIX1         ((unsigned char) 0xe0)
#define PREFIX2         ((unsigned char) 0xe1)

/* MODULGLOBALE VARIABLEN */

static char normal_tab[] =
 {
   0, 0, '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', 225, 39, 0,
   0, 'q', 'w', 'e', 'r', 't', 'z', 'u', 'i', 'o', 'p', 129, '+', 0,
   0, 'a', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 148, 132, '^', 0, '#',
   'y', 'x', 'c', 'v', 'b', 'n', 'm', ',', '.', '-', 0,
   '*', 0, ' ', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, '-',
   0, 0, 0, '+', 0, 0, 0, 0, 0, 0, 0, '<', 0, 0
 };

static char shift_tab[] =
 {
   0, 0, '!', '"', 21, '$', '%', '&', '/', '(', ')', '=', '?', 96, 0,
   0, 'Q', 'W', 'E', 'R', 'T', 'Z', 'U', 'I', 'O', 'P', 154, '*', 0,
   0, 'A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 153, 142, 248, 0, 39,
   'Y', 'X', 'C', 'V', 'B', 'N', 'M', ';', ':', '_', 0,
   0, 0, ' ', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, '>', 0, 0
 };

static char alt_tab[] =
 {
   0, 0, 0, 253, 0, 0, 0, 0, '{', '[', ']', '}', '\\', 0, 0,
   0, '@', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, '~', 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 230, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, '|', 0, 0
 };

/* IMPLEMENTIERUNG DER EXPORTIERTEN FUNKTIONEN */

/* REBOOT: Fuehrt ein Reset durch und loest damit ein Reboot aus.            */

void reboot ()
 {
   int status;

   /* Dem BIOS mitteilen, dass das Reset beabsichtigt war und kein */
   /* Speichertest durchgefuehrt werden muss. */

   *(unsigned short*) 0x472 = 0x1234;

   /* Der Tastaturcontroller soll das Reset ausloesen. */
   do
     { status = inb (KEYBD_CTRL_PORT);
     } while ((status & INPB) != 0);
   outb (KEYBD_CTRL_PORT, CMD_RESET);    /* Reset */
}

/* SET_REPEAT_RATE: Funktion zum Einstellen der Wiederholungsrate der        */
/*                  Tastatur.                                                */

void set_repeat_rate (int speed, int delay)
{
  int status;

  /* Wiederholrate der Tastatur einstellen */
   forbid_int (HW_INT_KEYBD);
   outb (KEYBD_DATA_PORT, CMD_SET_SPEED);
   do
     { status = inb (KEYBD_CTRL_PORT);
     } while ((status & INPB) != 0);
   status = inb (KEYBD_DATA_PORT); /* Ack */
   outb (KEYBD_DATA_PORT, ((delay & 3) << 5) | (speed & 63));
   allow_int (HW_INT_KEYBD);
}

/* DECODE: Interpretiert die Make und Break-Codes der Tastatur und liefert   */
/*         den ASCII Code, den Scancode und Informationen darueber, welche   */
/*         zusaetzlichen Tasten wie Shift und Ctrl gedrueckt wurden.         */
/*         Ein Rueckgabewert != 0 bedeutet, dass das Zeichen komplett ist,   */
/*         anderenfalls fehlen noch Make oder Breakcodes.                    */

int decode (unsigned char key, 
	    char* ascii, char* scancode, char* modifiers)
 {
   static unsigned char modifier_keys = MBIT_NUM_LOCK;
   static unsigned char prefix = 0;


   if (key == PREFIX1 || key == PREFIX2)
     { 
       prefix = (char) key;
       return 0;
     }
   
   if (key & BREAK)
     {
       key &= ~BREAK;
       switch (key)
	 {
	 case 69:
	   if (prefix == PREFIX2)
	     {
	       *ascii     = 0;
	       *scancode  = (char)key;
	       *modifiers = modifier_keys;
	       prefix     = 0;
	       return 1;
	     }
	   else
	     {
	       modifier_keys ^= MBIT_NUM_LOCK;
	     }
	   break;
	 case 58:
	   modifier_keys ^= MBIT_CAPS_LOCK;
	   break;
	 case 70:
	   modifier_keys ^= MBIT_SCROLL_LOCK;
	   break;
	 case 42:  
	 case 54:
	   modifier_keys &= ~MBIT_SHIFT;
	   break;
	 case 56:
	   if (prefix == PREFIX1)
	     modifier_keys &= ~MBIT_ALT_RIGHT;
	   else
	     modifier_keys &= ~MBIT_ALT_LEFT;
	   break;
	 case 29:
	   if (prefix == PREFIX1)
	     modifier_keys &= ~MBIT_CTRL_RIGHT;
	   else
	     modifier_keys &= ~MBIT_CTRL_LEFT;
	   break;
	 }
       return 0;
     }

   switch (key)
     {
     case 42:  
     case 54:
       modifier_keys |= MBIT_SHIFT;
       break;
     case 56:
       if (prefix == PREFIX1)
	 modifier_keys |= MBIT_ALT_RIGHT;
       else
	 modifier_keys |= MBIT_ALT_LEFT;
       break;
     case 29:
       if (prefix == PREFIX1)
	 modifier_keys |= MBIT_CTRL_RIGHT;
       else
	 modifier_keys |= MBIT_CTRL_LEFT;
       break;
     default:
       if (modifier_keys & (MBIT_SHIFT|MBIT_CAPS_LOCK))
	 *ascii = shift_tab[key];
       else if (modifier_keys & MBIT_ALT_RIGHT)
	 *ascii = alt_tab[key];
       else
	 *ascii = normal_tab[key];
       *scancode  = (char)key;
       *modifiers = modifier_keys;
       prefix     = 0;
       return 1;
     }
   return 0;
 }
