#include "objects/list.h"
#include "device/keybd.h"
#include "machine/pic.h"
#include "machine/keyctrl.h"
#include "machine/ioport.h"
#include "machine/plug.h"
#include "machine/screen.h"
#include "thread/process.h"
#include "guard/lock.h"
#include "guard/sema.h"
#include "guard/syscall.h"

#define KEYB_BUFFER_SIZE	16

static char	keyb_buffer[KEYB_BUFFER_SIZE*2];
static int	keyb_buffer_start;
static int	keyb_buffer_end;

static struct epilogue_descriptor keybd_epi;

static unsigned char keyb_data[KEYB_BUFFER_SIZE];

static int 	keyb_data_start;
static int	keyb_data_end;

static struct semaphore keyboard_sema;


int getch() {
	int act = 0x00;
	
	if ( keyb_buffer_start == keyb_buffer_end )
		return 0x00;
		
	act = keyb_buffer[keyb_buffer_start];
	act |= (keyb_buffer[keyb_buffer_start+1] << 0x08);
	keyb_buffer_start += 0x02;
	keyb_buffer_start &= (KEYB_BUFFER_SIZE*2-1);

	return act;
}



int getchar() {
	sema_p(&keyboard_sema);
	return getch();
}


void init_keyboard(void)
{
	plug_assign(keyboard_prologue, VEC_KEYBD);
	set_repeat_rate(1,1);
	allow_int(HW_INT_KEYBD);
	keyb_buffer_start = keyb_buffer_end = 0;

	keybd_epi.epilogue = keyboard_epilogue;
	keybd_epi.queued = 0;
	
	keyb_data_start = keyb_data_end = 0x00;
	
	sys_sema_init(&keyboard_sema,0);
}

void keyboard_prologue(void)
{
	unsigned char keycode = inb(KEYBD_DATA_PORT);
	
	if ( !(((keyb_data_end+1) & (KEYB_BUFFER_SIZE-1)) == keyb_data_start) ) {
		keyb_data[keyb_data_end++] = keycode;	/* nur wenn ringpuffer leer */
		keyb_data_end &= KEYB_BUFFER_SIZE-1;
	}
	
	ack();
	relay(&keybd_epi);
			
}

void keyboard_epilogue(void)
{
	char	ascii;
	char	scancode;
	char	modifiers;
	unsigned char keycode;
	
	if ( keyb_data_start == keyb_data_end )		/* ringpuffer leer? */
		return;
		
	keycode = keyb_data[keyb_data_start++];
	keyb_data_start &= KEYB_BUFFER_SIZE - 1;

	keycode = decode(keycode,&ascii,&scancode,&modifiers);
	
	if ( (scancode == SCAN_DELETE) &&
		 (modifiers & (MBIT_ALT_LEFT | MBIT_ALT_RIGHT)) &&
		 (modifiers & (MBIT_CTRL_LEFT | MBIT_CTRL_RIGHT)) )
		reboot();
		
	if ( keycode ) {
		
		if ( !(((keyb_buffer_end + 0x02) & (KEYB_BUFFER_SIZE*2-1)) == keyb_buffer_start) ) {
			keyb_buffer[keyb_buffer_end] = ascii;
			keyb_buffer[keyb_buffer_end+1] = scancode;
			keyb_buffer_end += 0x02;
			keyb_buffer_end &= (KEYB_BUFFER_SIZE*2-1);
			sema_v(&keyboard_sema);
		}

	}
}
